package cn.com.servyou.ca.service.impl;

import cn.com.servyou.ca.dto.ActiveRequestDTO;
import cn.com.servyou.ca.dto.ActiveResultDTO;
import cn.com.servyou.ca.dto.CaConfigDTO;
import cn.com.servyou.ca.dto.CaResultDTO;
import cn.com.servyou.ca.dto.CertRequestDTO;
import cn.com.servyou.ca.dto.CertResultDTO;
import cn.com.servyou.ca.dto.RandomDTO;
import cn.com.servyou.ca.dto.SignRequestDTO;
import cn.com.servyou.ca.dto.SignResultDTO;
import cn.com.servyou.ca.request.CaRequest;
import cn.com.servyou.ca.service.CaService;
import cn.com.servyou.ca.service.SignService;
import cn.com.servyou.ca.utils.CASecurityUtil;
import cn.com.servyou.constant.ServyouKeyConstant;
import cn.com.servyou.rmi.client.ClientProxyFactory;
import cn.com.servyou.utils.JSONUtil;
import cn.com.servyou.utils.LoggerUtil;
import com.sheca.safeengine.javasafeengine;
import lombok.extern.slf4j.Slf4j;
import org.bouncycastle.util.encoders.HexEncoder;

import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.URLDecoder;
import java.net.URLEncoder;
import java.time.LocalDateTime;
import java.time.format.DateTimeFormatter;
import java.util.Random;

/**
 * @author zhouww
 * @since 20211028
 */
@Slf4j
public class CaServiceImpl implements CaService {

    private ClientProxyFactory clientProxyFactory;

    public CaServiceImpl(ClientProxyFactory clientProxyFactory) {
        this.clientProxyFactory = clientProxyFactory;
    }

    @Override
    public String getCA(String cert, String operatorName, String cardNo, String socialSecurity, int inKeyIndex) {
        try {
            CaRequest caRequest = clientProxyFactory.getCacheRequest(CaRequest.class);
            // 激活
            String time = getCurrentTime();
            String sign = CASecurityUtil.signature(getActivePrivateKey(), time);
            ActiveRequestDTO activeRequestDTO = new ActiveRequestDTO();
            activeRequestDTO.setOpratorName(operatorName);
            activeRequestDTO.setCardNo(cardNo);
            activeRequestDTO.setTime(time);
            activeRequestDTO.setSign(URLEncoder.encode(sign, ServyouKeyConstant.ENCODING_UTF8));
            activeRequestDTO.setSocialSecurity(socialSecurity);
            ActiveResultDTO activeResultDTO = caRequest.active(activeRequestDTO);
            if (0 != activeResultDTO.getRet()) {
                throw new RuntimeException(activeResultDTO.getMsg());
            }

            // 获取凭证
            String time2 = getCurrentTime();
            String sign2 = CASecurityUtil.signature(getCertAndSignPrivateKey(), time2);
            sign2 = URLEncoder.encode(sign2, ServyouKeyConstant.ENCODING_UTF8);
            String tempCert = URLEncoder.encode(cert, ServyouKeyConstant.ENCODING_UTF8);
            CertRequestDTO certRequestDTO = new CertRequestDTO();
            certRequestDTO.setCert(tempCert);
            certRequestDTO.setTime(time2);
            certRequestDTO.setSign(sign2);
            certRequestDTO.setAlg("RSA");
            CertResultDTO certResultDTO = caRequest.cert(certRequestDTO);
            String userPublicKey = certResultDTO.getData().getCert();
            userPublicKey = URLDecoder.decode(userPublicKey, ServyouKeyConstant.ENCODING_UTF8);
            byte[] bCert = javasafeengine.b64tohex(userPublicKey);
            String commonId = javasafeengine.getCertInfoByOID("1.2.156.112570.11.205", bCert);
            String socialSecurityFromUsePublicKey = javasafeengine.getCertInfoByOID("1.2.156.10260.4.1.2", bCert);
            if (!socialSecurity.equals(socialSecurityFromUsePublicKey)) {
                throw new RuntimeException("激活证书验证社会保险号失败");
            }

            // 获取随机码签名
            String random = getRandom(16);
            SignService signService = clientProxyFactory.getSignService();
            String signRandom = signService.signRandom(random, inKeyIndex);
            String time3 = getCurrentTime();
            String sign3 = CASecurityUtil.signature(getCertAndSignPrivateKey(), time3);
            sign3 = URLEncoder.encode(sign3, ServyouKeyConstant.ENCODING_UTF8);

            // 用户数据签名
            RandomDTO randomDTO = new RandomDTO();
            randomDTO.setRandom(random);
            randomDTO.setCommonId(commonId);
            randomDTO.setSocialSecurity(socialSecurityFromUsePublicKey);

            String userSignPlan = JSONUtil.toJSON(randomDTO);
            String hexEncodeUserSign = hexEncode(userSignPlan);

            SignRequestDTO signRequestDTO = new SignRequestDTO();
            signRequestDTO.setData(hexEncodeUserSign);
            signRequestDTO.setRandom(random);
            signRequestDTO.setRandomSign(signRandom);
            signRequestDTO.setCert(cert);
            signRequestDTO.setTime(time3);
            signRequestDTO.setSign(sign3);
            signRequestDTO.setSignAlg("SM3withSM2");
            SignResultDTO signResultDTO = caRequest.sign(signRequestDTO);
            if (0 != signResultDTO.getRet()) {
                throw new RuntimeException(signResultDTO.getMsg());
            }

            String userSign = signResultDTO.getData().getSign();
            userSign = URLDecoder.decode(userSign, ServyouKeyConstant.ENCODING_UTF8);
            String lastCert = signResultDTO.getData().getCert();
            lastCert = URLDecoder.decode(lastCert, ServyouKeyConstant.ENCODING_UTF8);
            byte[] lastCertArr = javasafeengine.b64tohex(lastCert);
            String lastSocialSecurity = javasafeengine.getCertInfoByOID("1.2.156.10260.4.1.2", lastCertArr);
            if (!lastSocialSecurity.equals(socialSecurity)) {
                throw new RuntimeException("签名时验证社会保险号失败");
            }

            CaResultDTO caResultDTO = new CaResultDTO();
            caResultDTO.setUserCert(lastCert);
            caResultDTO.setUserSign(userSign);
            caResultDTO.setRandom(JSONUtil.toJSON(randomDTO));
            caResultDTO.setActionType("verify");
            caResultDTO.setLoginType("ukey");
            caResultDTO.setAppType(2);
            return JSONUtil.toJSON(caResultDTO);
        } catch (Exception e) {
            LoggerUtil.error(log, e, "获取ca失败");
            throw new RuntimeException(e);
        }
    }

    @Override
    public String signData(String cert, int inKeyIndex, String data) {
        try {
            CaRequest caRequest = clientProxyFactory.getCacheRequest(CaRequest.class);

            String hexEncodeUserSign = hexEncode(data);
            String random = getRandom(16);
            SignService signService = clientProxyFactory.getSignService();
            String signRandom = signService.signRandom(random, inKeyIndex);
            String time = getCurrentTime();
            String timeSign = CASecurityUtil.signature(getCertAndSignPrivateKey(), time);
            timeSign = URLEncoder.encode(timeSign, ServyouKeyConstant.ENCODING_UTF8);

            SignRequestDTO signRequestDTO = new SignRequestDTO();
            signRequestDTO.setData(hexEncodeUserSign);
            signRequestDTO.setRandom(random);
            signRequestDTO.setRandomSign(signRandom);
            signRequestDTO.setCert(cert);
            signRequestDTO.setTime(time);
            signRequestDTO.setSign(timeSign);
            signRequestDTO.setSignAlg("SM3withSM2");
            SignResultDTO signResultDTO = caRequest.sign(signRequestDTO);
            if (0 != signResultDTO.getRet()) {
                throw new RuntimeException(signResultDTO.getMsg());
            }
            String dateSign = signResultDTO.getData().getSign();
            return URLDecoder.decode(dateSign, ServyouKeyConstant.ENCODING_UTF8);
        } catch (Exception e) {
            LoggerUtil.error(log, e, "数据签名失败");
            throw new RuntimeException(e);
        }
    }

    private String hexEncode(String data) {
        try (ByteArrayOutputStream baos = new ByteArrayOutputStream(10240)) {
            byte[] userSignByteArr = data.getBytes(ServyouKeyConstant.ENCODING_UTF8);
            int size = new HexEncoder().encode(userSignByteArr, 0, userSignByteArr.length, baos);
            byte[] signData = baos.toByteArray();
            return new String(signData, 0, size);
        } catch (IOException e) {
            LoggerUtil.error(log, e, "签名数据组装");
            throw new RuntimeException(e);
        }
    }

    private String getRandom(int size) {
        StringBuilder stringBuilder = new StringBuilder();
        Random random = new Random();
        for (int i = 0; i < size; i++) {
            stringBuilder.append(random.nextInt(10));
        }
        return stringBuilder.toString();
    }

    private static String getCurrentTime() {
        LocalDateTime startDateTime = LocalDateTime.now();
        DateTimeFormatter formatter = DateTimeFormatter.ofPattern("yyyyMMddHHmmss");
        return formatter.format(startDateTime);
    }

    private String getActivePrivateKey() {
        CaConfigDTO caConfigDTO = getCaConfigDTO();
        if (null != caConfigDTO) {
            return caConfigDTO.getCaActivePrivateKey();
        }
        return null;
    }

    private String getCertAndSignPrivateKey() {
        CaConfigDTO caConfigDTO = getCaConfigDTO();
        if (null != caConfigDTO) {
            return caConfigDTO.getCertAndSignPrivateKey();
        }
        return null;
    }

    private CaConfigDTO getCaConfigDTO() {
        if (null == clientProxyFactory) {
            return null;
        }
        return clientProxyFactory.getCaConfigDTO();
    }
}
