package cn.com.servyou.rmi.client;

import cn.com.servyou.ca.enums.UrlMatchEnum;
import cn.com.servyou.ca.service.CaService;
import cn.com.servyou.dto.ApiResponse;
import cn.com.servyou.rmi.customization.annotations.FillCA;
import cn.com.servyou.rmi.customization.annotations.ServyouSignature;
import cn.com.servyou.rmi.http.FormUrlEncodedRequestBody;
import cn.com.servyou.rmi.http.JSONRequestBody;
import cn.com.servyou.rmi.http.PathVariable;
import cn.com.servyou.rmi.http.RequestHeader;
import cn.com.servyou.rmi.http.RequestMapping;
import cn.com.servyou.rmi.http.RequestMethod;
import cn.com.servyou.rmi.http.RequestParam;
import cn.com.servyou.utils.ArrayUtil;
import cn.com.servyou.utils.EncryptionUtil;
import cn.com.servyou.utils.JSONUtil;
import cn.com.servyou.utils.LoggerUtil;
import cn.com.servyou.utils.MapUtil;
import cn.com.servyou.utils.StringUtil;
import com.alibaba.fastjson.JSONArray;
import com.alibaba.fastjson.JSONObject;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.ObjectNode;
import com.fasterxml.jackson.databind.type.TypeFactory;
import lombok.extern.slf4j.Slf4j;
import org.springframework.util.StringUtils;
import org.springframework.web.bind.annotation.RequestBody;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Field;
import java.lang.reflect.InvocationHandler;
import java.lang.reflect.Method;
import java.lang.reflect.ParameterizedType;
import java.lang.reflect.Type;
import java.lang.reflect.TypeVariable;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;
import java.util.concurrent.Semaphore;

import static cn.com.servyou.constant.ServyouKeyConstant.APP_KEY;
import static cn.com.servyou.constant.ServyouKeyConstant.ENCODING_UTF8;
import static cn.com.servyou.constant.ServyouKeyConstant.SIGNATURE;
import static cn.com.servyou.constant.ServyouKeyConstant.STRING_GATEWAY;
import static cn.com.servyou.constant.ServyouKeyConstant.TIMESTAMP;
import static cn.com.servyou.constant.ServyouKeyConstant.VERSION;
import static cn.com.servyou.constant.ServyouKeyConstant.X_REQUEST_NONCE;
import static cn.com.servyou.constant.SymbolConstant.AND;
import static cn.com.servyou.constant.SymbolConstant.DASH;
import static cn.com.servyou.constant.SymbolConstant.EQUAL;
import static cn.com.servyou.constant.SymbolConstant.QUESTION_MARK;

/**
 * 接口代理的上下文
 * 功能有限只解析部分的注解
 * ServiceClient
 * RequestBody
 * PathVariable
 * RequestParam
 * RequestHeader
 *
 * @author zhouww
 * @since 20191024
 */
@Slf4j
public class ClientProxy implements InvocationHandler {

    /**
     * 接口类
     */
    private Class serviceClientClazz;

    /**
     * CA操作并发控制
     */
    private Semaphore semaphore = new Semaphore(1, true);

    /**
     * 应用工厂类，需要获取配置信息
     */
    private ClientProxyFactory clientProxyFactory;

    private static final List<String> ORIGINAL_TYPE_NAME_LIST = new ArrayList<>();

    static {
        ORIGINAL_TYPE_NAME_LIST.add(String.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Integer.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Double.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Float.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Character.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Short.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(Byte.class.getName());

        ORIGINAL_TYPE_NAME_LIST.add(String.class.getName());
        ORIGINAL_TYPE_NAME_LIST.add(String.class.getName());

    }


    public ClientProxy(Class serviceClientClazz, ClientProxyFactory clientProxyFactory) {
        this.serviceClientClazz = serviceClientClazz;
        this.clientProxyFactory = clientProxyFactory;
    }

    @Override
    public Object invoke(Object proxy, Method method, Object[] args) throws Throwable {
        ServiceClient serviceClient = (ServiceClient) serviceClientClazz.getAnnotation(ServiceClient.class);
        if (null == serviceClient) {
            LoggerUtil.info(log, "此接口没有ServiceClient注解 serviceName={0}", serviceClientClazz.getName());
            return new RuntimeException("接口没有ServiceClient注解");
        }

        RequestMapping requestMapping = method.getAnnotation(RequestMapping.class);

        if (null == requestMapping) {
            LoggerUtil.info(log, "requestConfig serviceName={0}, methodName={1}", serviceClientClazz.getName(), method.getName());
            throw new RuntimeException("请定义调用的方法取消路径注解");
        }

        RequestMethod requestMethod = requestMapping.method();

        // 解析请求相关的参数
        String clientUrl = splicingUrl(serviceClient, requestMapping, method, args);
        Map<String, String> headerMap = getRequestHeader(requestMapping, method, args);
        String body = getBody(method, args);

        ServyouSignature classServyouSignature = (ServyouSignature) serviceClientClazz.getAnnotation(ServyouSignature.class);
        ServyouSignature methodServyouSignature = method.getAnnotation(ServyouSignature.class);
        if (null != classServyouSignature || null != methodServyouSignature) {
            String timestamp = System.currentTimeMillis() + "";
            String version = "1.0.0";
            String xReqNonce = UUID.randomUUID().toString().replaceAll(DASH, "");
            String appKey = clientProxyFactory.getAppKey();
            String appSecret = clientProxyFactory.getAppSecret();

            headerMap.put(TIMESTAMP, timestamp);
            headerMap.put(VERSION, version);
            headerMap.put(X_REQUEST_NONCE, xReqNonce);
            headerMap.put(APP_KEY, appKey);
            if (StringUtil.isNoneBlank(appKey, appSecret)) {
                // 只有在appKey和appSecret都不为空的情况下进行签名
                Map<String, String> queryStringMap = getQueryStringMap(clientUrl);
                String signature = "";
                if (StringUtil.isNoneBlank(appKey, appSecret)) {
                    signature = EncryptionUtil.parameterEncryption(appKey, appSecret, timestamp, version, xReqNonce, queryStringMap);
                }
                headerMap.put(SIGNATURE, signature);
            }
        }

        if (log.isDebugEnabled()){
            LoggerUtil.info(log, "[代理请求]请求参数 method={0}, url={1}, header={2}, body={3}", requestMethod, clientUrl, JSONUtil.toJSON(headerMap), body);
        } else {
            LoggerUtil.info(log, "[代理请求]请求参数 method={0}, url={1}, header={2}", requestMethod, clientUrl, JSONUtil.toJSON(headerMap));
        }
        byte[] bytes = clientProxyFactory.request(requestMethod, clientUrl, headerMap, body);
        String jsonResult = null;
        if (ArrayUtil.isNotEmpty(bytes)) {
            jsonResult = new String(bytes, ENCODING_UTF8);
            if (Boolean.TRUE.equals(clientProxyFactory.getLogSwitchOpen())) {
                LoggerUtil.info(log, "请求的结果 bizNo={0} result={1}", body, jsonResult);
            }
            //出参解密操作
            if (!Objects.isNull(this.clientProxyFactory.getBaseMessageHandler()) && this.clientProxyFactory.getBaseMessageHandler().getConfiguration().isDecryptOutData()) {
                jsonResult = this.clientProxyFactory.getBaseMessageHandler().decrypt(jsonResult);
                LoggerUtil.info(log, "解密后的出参={0}", jsonResult);
            }
        }

        Type returnType = method.getGenericReturnType();
        JavaType javaType = getJavaType(returnType);
        Class returnClass = method.getReturnType();
        String className = returnClass.getName();

        if ("void".equalsIgnoreCase(className)) {
            return null;
        } else if (StringUtil.isBlank(jsonResult)) {
            // 没有数据的情况下返回空
            return null;
        } else if (String.class.getName().equalsIgnoreCase(className)) {
            return jsonResult;
        } else {
            try {
                return JSONUtil.parseJSON(jsonResult, javaType);
            } catch (Exception e) {
                // 如果反序列化报错，则只序列化首部。兼容body返回字符串的场景(这个通用网关返回的数据和应用不一致导致的)，
                // 为所有的返回数据格式都是封装在ApiResponse中，所以直接只用具体的类（这样设计不好，具体的应用影响到了请求的框架，但是这样能做兼容，并且可以设置body为null。避免客户调用接口获取boyd的时候报异常）
                ApiResponse apiResponse = JSONUtil.parseJSON(jsonResult, ApiResponse.class);
                apiResponse.setBody(null);
                return apiResponse;
            }
        }
    }

    /**
     * 填充ca信息
     * 只支持 {@link JSONRequestBody}注解的对象或者json格式字符串
     *
     * @param bodyJSON 序列化后的结果
     * @param method   method
     */
    private String fillCAParam(String bodyJSON, Method method) {
        // 填充ca
        FillCA classFillCA = (FillCA) serviceClientClazz.getAnnotation(FillCA.class);
        FillCA methodFillCA = method.getAnnotation(FillCA.class);
        if (null == classFillCA && null == methodFillCA) {
            // 没有FillCA注解不处理填充CA的逻辑
            return bodyJSON;
        }
        // 单位编号 = 社会保险号
        Map<String, String> keyMap = JSONUtil.getField(bodyJSON, "cazs", "allowFillCA", "operatorCert", "operatorName", "cardNo", "dwbh", "inKeyIndex", "socialSecurity");
        String allowFillCA = keyMap.get("allowFillCA");
        if (StringUtil.isBlank(allowFillCA) || "false".equals(allowFillCA)) {
            return bodyJSON;
        }
        // ca
        String cazs = getCAInfo(keyMap, bodyJSON);
        String socialSecurity = getSocialSecurity(keyMap);
        if (StringUtil.isBlank(cazs)) {
            // 填充ca信息
            CaService caService = clientProxyFactory.getCAService();
            // String类型数据，从String中获取参数，并且将ca数据填充回CA
            String cert = keyMap.get("operatorCert");
            String operatorName = keyMap.get("operatorName");
            String cardNo = keyMap.get("cardNo");
            String inKeyIndexStr = keyMap.get("inKeyIndex");
            if (StringUtil.isBlank(inKeyIndexStr)) {
                throw new RuntimeException("请输入索引号");
            }
            int inKeyIndex = Integer.valueOf(inKeyIndexStr);
            // 获取CA进行并发控制
            semaphore.acquireUninterruptibly();
            try {
                String ca = caService.getCA(cert, operatorName, cardNo, socialSecurity, inKeyIndex);
                String result = JSONUtil.removeField(bodyJSON, "allowFillCA", "operatorCert", "operatorName", "cardNo", "inKeyIndex", "socialSecurity");
                bodyJSON = setCAInfo(result, ca);
                bodyJSON = JSONUtil.parseArray(bodyJSON, getSignArrayList(method), jsonNode -> {
                    if (null == jsonNode || jsonNode.isNull() || !jsonNode.isObject()) {
                        return;
                    }
                    JsonNode newJson = jsonNode.get("tszdxx");
                    if (newJson != null) {
                        jsonNode = newJson;
                    }
                    JsonNode businessDigestNode = jsonNode.get("sjzy");
                    String businessDigest = "";
                    if (null != businessDigestNode || !businessDigestNode.isTextual()) {
                        businessDigest = businessDigestNode.asText();
                    }
                    if (StringUtil.isBlank(businessDigest)) {
                        return;
                    }
                    JsonNode digestNode = jsonNode.get("sjqmlb");
                    if (null == digestNode || !digestNode.isArray()) {
                        return;
                    }
                    List<String> digestSignList = new ArrayList<>();
                    for (JsonNode subDigestNode : digestNode) {
                        String digest = subDigestNode.asText();
                        int index = digest.indexOf("|");
                        if (index <= 0) {
                            throw new RuntimeException("数据签名不符合规定格式");
                        }
                        String prefix = digest.substring(0, ++index);
                        String suffix = digest.substring(index);
                        String digestSign = prefix + (caService.signData(cert, inKeyIndex, suffix));
                        digestSignList.add(digestSign);
                    }
                    ObjectNode objectNode = (ObjectNode) jsonNode;
                    ObjectMapper objectMapper = new ObjectMapper();
                    try {
                        JsonNode digestSignListNode = objectMapper.readTree(JSONObject.toJSONString(digestSignList));
                        if (log.isDebugEnabled()) {
                            LoggerUtil.info(log, "签名前的参数 digestNode={0}， 签名后的参数 digestSignListNode={1}",
                                    JSONUtil.toJSON(digestNode), JSONUtil.toJSON(digestSignListNode));
                        }
                        objectNode.replace("sjqmcaqmlb", digestSignListNode);
                    } catch (IOException e) {
                        LoggerUtil.error(log, e, "设置签名参数列表失败 digestSignList={1}",
                                JSONUtil.toJSON(digestNode), JSONUtil.toJSON(digestSignList));
                    }
                });
            } finally {
                semaphore.release();
            }
        }
        return bodyJSON;
    }

    /**
     * 获取单位编号
     *
     * @param keyMap
     * @return
     */
    public static String getSocialSecurity(Map<String, String> keyMap) {
        String socialSecurity = keyMap.get("socialSecurity");
        if (StringUtil.isBlank(socialSecurity)) {
            socialSecurity = keyMap.get("dwbh");
        }
        return socialSecurity;
    }

    /**
     * 获取ca证书
     *
     * @param keyMap
     * @param bodyJSON
     * @return
     */
    public static String getCAInfo(Map<String, String> keyMap, String bodyJSON) {
        JSONObject jsonObject = JSONObject.parseObject(bodyJSON);
        JSONArray jsonArray = jsonObject.getJSONArray("ywblzhlb");
        // 业务办理账号列表为空取外部的CAZS
        if (jsonArray == null) {
            return keyMap.get("cazs");
        }
        for (int i = 0; i < jsonArray.size(); i++) {
            JSONObject object = jsonArray.getJSONObject(i);
            Object insuranceCategory = object.get("blxm");
            if (insuranceCategory == null) {
                continue;
            }
            if (Objects.equals(insuranceCategory.toString(), "社保")) {
                Object cazs = object.get("cazs");
                return cazs == null ? null : cazs.toString();
            }
        }
        return null;
    }

    /**
     * 设置ca证书
     *
     * @param bodyJSON
     * @param ca
     * @return
     */
    public static String setCAInfo(String bodyJSON, String ca) {
        JsonNode jsonNode = JSONUtil.readTree(bodyJSON);
        JsonNode subNode = jsonNode.get("ywblzhlb");
        if (subNode == null || !subNode.isArray()) {
            bodyJSON = JSONUtil.addField(bodyJSON, "cazs", ca);
            return bodyJSON;
        }
        ArrayNode arrayNode = (ArrayNode) subNode;
        for (JsonNode node : arrayNode) {
            JsonNode insuranceCategory = node.get("blxm");
            if (insuranceCategory == null) {
                continue;
            }
            if (Objects.equals(insuranceCategory.asText(), "社保")) {
                ObjectNode objectNode = (ObjectNode) node;
                objectNode.put("cazs", ca);
                return JSONUtil.writeValueAsString(jsonNode);
            }
        }
        return bodyJSON;
    }

    /**
     * 获取签名参数路径
     *
     * @param method 方法
     * @return
     */
    private List<String> getSignArrayList(Method method) {
        RequestMapping requestMapping = method.getAnnotation(RequestMapping.class);
        if (null == requestMapping) {
            return Collections.emptyList();
        }
        String urlPath = requestMapping.value();
        if (StringUtils.isEmpty(urlPath)) {
            return Collections.emptyList();
        }
        UrlMatchEnum urlMatchEnum = UrlMatchEnum.matchByUrl(urlPath);
        if (urlMatchEnum == null) {
            return Collections.emptyList();
        }
        return Arrays.asList(urlMatchEnum.getParamName());
    }

    /**
     * 必须指定所有的范型，
     * 比如ApiResponse不指定范型，则会报错
     *
     * @param type
     * @return
     */
    public static JavaType getJavaType(Type type) {
        //判断是否带有泛型
        if (type instanceof ParameterizedType) {
            Type[] actualTypeArguments = ((ParameterizedType) type).getActualTypeArguments();
            //获取泛型类型
            Class rowClass = (Class) ((ParameterizedType) type).getRawType();
            JavaType[] javaTypes = new JavaType[actualTypeArguments.length];

            for (int i = 0; i < actualTypeArguments.length; i++) {
                //泛型也可能带有泛型，递归获取
                javaTypes[i] = getJavaType(actualTypeArguments[i]);
            }
            return TypeFactory.defaultInstance().constructParametricType(rowClass, javaTypes);
        } else {
            //简单类型直接用该类构建JavaType
            Class cla = (Class) type;
            TypeVariable<?>[] vars = cla.getTypeParameters();
            if (vars != null && vars.length != 0) {
                // 有范型但是没有指定特定类型
                return TypeFactory.defaultInstance().constructType(cla);
            } else {
                return TypeFactory.defaultInstance().constructParametricType(cla, new JavaType[0]);
            }
        }
    }

    /**
     * 对于本地的请求则做过滤gateway处理
     * 如果是完整的URL则过滤第一个路径 http://localhost:8080/gateway/a/b
     * 如果是部分的路径则过滤前置路径 /gateway/a/b
     *
     * @param url 请求URL
     * @return
     */
    private String decorateGateway(String url){
        if (StringUtil.isBlank(url)) {
            return url;
        }
        try {
            URL urlObj = new URL(url);
            // 完整URL路径
            int port = urlObj.getPort();
            String queryStr = urlObj.getQuery();
            String path = urlObj.getPath();

            String resultUrl = urlObj.getProtocol() + "://" + urlObj.getHost();
            if (port > 0) {
                resultUrl += ":" + port;
            }

            if (StringUtil.isNotBlank(path)) {
                if (ClientSwitch.gatewayRemoveSwitch && path.startsWith(STRING_GATEWAY, 1)) {
                    path = path.substring(STRING_GATEWAY.length() + 1);
                    if (!StringUtils.isEmpty(ClientSwitch.pathPrefix)) {
                        path = ClientSwitch.pathPrefix + path;
                    }
                }
                resultUrl += path;
            }

            if (StringUtil.isNotBlank(queryStr)) {
                resultUrl += "?" + queryStr;
            }

            return resultUrl;
        } catch (MalformedURLException e) {
            // 非完整URL路径
            url = url.trim();
            if (url.startsWith("/gateway")) {
                url = url.substring(8);
            } else if (url.startsWith("gateway")) {
                url = url.substring(7);
            }
            if (!StringUtils.isEmpty(ClientSwitch.pathPrefix)) {
                url = ClientSwitch.pathPrefix + url;
            }
        }
        return url;
    }

    /**
     * 拼接URL
     *
     * @param serviceClient serviceClient
     * @param requestConfig serviceClient
     * @return 拼接URL
     */
    private String splicingUrl(ServiceClient serviceClient, RequestMapping requestConfig, Method method, Object[] params) {
        String clientUrl = serviceClient.url();
        String serverName = serviceClient.value();
        if (StringUtil.isAllBlank(clientUrl, serverName)) {
            LoggerUtil.info(log, "ServiceClient的value和URL不能都为空");
            throw new RuntimeException("ServiceClient的value和URL不能都为空");
        }
        if (StringUtil.isBlank(clientUrl)) {
            clientUrl = clientProxyFactory.getHost(serverName);
        }

        String path = requestConfig.value();
        if (ClientSwitch.gatewayRemoveSwitch) {
            path = decorateGateway(path);   // 只处理RequestMapping注解的前缀
        }
        if (!clientUrl.endsWith("/") && !path.startsWith("/")) {
            clientUrl += "/" + path;
        } else  {
            clientUrl += path;
        }

        // 替换路径参数
        Annotation[][] paramAnnotations = method.getParameterAnnotations();
        for (int i = 0; i < paramAnnotations.length; i++) {
            Annotation[] annotations = paramAnnotations[i];
            if (ArrayUtil.isEmpty(annotations)) {
                continue;
            }
            for (Annotation annotation : annotations) {
                if (annotation instanceof PathVariable) {
                    PathVariable pathVariable = (PathVariable) annotation;
                    String key = pathVariable.value();
                    String value = null != params[i] ? params[i].toString() : null;
                    clientUrl.replaceAll("{" + key + "}", value);
                } else if (annotation instanceof RequestParam) {
                    RequestParam requestParam = (RequestParam) annotation;
                    String key = requestParam.value();
                    String value = null != params[i] ? params[i].toString() : null;
                    clientUrl = appendQueryString(clientUrl, key, value);
                }
            }
        }
        return clientUrl;
    }

    public String appendQueryString(String url, String queryKey, String queryValue) {
        String queryParam = queryKey + EQUAL + queryValue;
        String linkCharacter = QUESTION_MARK;
        if (url.contains(QUESTION_MARK)) {
            linkCharacter = AND;
        }
        return url + linkCharacter + queryParam;
    }

    /**
     * 获取请求首部
     *
     * @param requestMapping requestMapping
     * @param method method
     * @param params params
     * @return 请求首部
     */
    private Map<String, String> getRequestHeader(RequestMapping requestMapping, Method method, Object[] params) {
        Map<String, String> headerMap = new HashMap<>();
        // 查找RequestMapping的Header配置
        String[] headers = requestMapping.headers();
        if (headers.length > 0) {
            for (String header : headers) {
                String key = header;
                String value = "";
                int index = header.indexOf(EQUAL);
                if (index > 0) {
                    // 不能没有等号 并且等号不能在第一位
                    key = header.substring(0, index);
                    value = header.substring(index + 1);
                }
                headerMap.put(key, value);
            }
        }

        // 查找参数中的Header配置
        Annotation[][] paramAnnotations = method.getParameterAnnotations();
        for (int i = 0; i < paramAnnotations.length; i++) {
            Annotation[] annotations = paramAnnotations[i];
            if (ArrayUtil.isEmpty(annotations)) {
                continue;
            }
            for (Annotation annotation : annotations) {
                if (annotation instanceof RequestHeader) {
                    RequestHeader pathVariable = (RequestHeader) annotation;
                    String key = pathVariable.value();
                    String value = JSONUtil.toJSON(params[i]);
                    headerMap.put(key, value);
                }
            }
        }

        return headerMap;
    }

    /**
     * 获取body
     *
     * @param method method
     * @param params params
     * @return 请求体
     */
    public String getBody(Method method, Object[] params) {
        // 查找参数中的Header配置
        Annotation[][] paramAnnotations = method.getParameterAnnotations();
        for (int i = 0; i < paramAnnotations.length; i++) {
            Annotation[] annotations = paramAnnotations[i];
            if (ArrayUtil.isEmpty(annotations)) {
                continue;
            }
            // 序列化
            for (Annotation annotation : annotations) {
                if (annotation instanceof JSONRequestBody) {
                    String jsonData = serializeBody(params[i]);
                    return fillCAParam(jsonData, method);
                }
                if (annotation instanceof RequestBody) {
                    return serializeBody(params[i]);
                } else if (annotation instanceof FormUrlEncodedRequestBody) {
                    return fromUrlEncoded(params[i]);
                }
            }
        }
        return null;
    }

    /**
     * 序列化参数
     *
     * @param param
     * @return
     */
    public String serializeBody(Object param) {
        if (null == param) {
            return null;
        }
        if (param instanceof String) {
            // 字符串 字节数组 不做序列化
            return (String) param;
        } else {
            return JSONUtil.toJSONExcludeNull(param);
        }
    }

    /**
     * from-url-encoded 格式的数据
     * 注意 只支持String类型的数据进行转化
     * TODO 扩展支持的类型
     *
     * @param param param
     * @return
     */
    public String fromUrlEncoded(Object param) {
        if (null == param) {
            return "";
        }
        Field[] fields = param.getClass().getDeclaredFields();
        if (ArrayUtil.isEmpty(fields)) {
            return "";
        }
        Map<String, String> paramMap = new HashMap<>();
        for (Field field : fields) {
            Class fieldType = field.getType();
            if (String.class.isAssignableFrom(fieldType)) {
                try {
                    field.setAccessible(true);
                    String fieldName = field.getName();
                    String value = (String) field.get(param);
                    if (null == value) {
                        value = "";
                    }
//                    else {
//                        try {
//                            value = URLEncoder.encode(value, ServyouKeyConstant.ENCODING_UTF8);
//                        } catch (UnsupportedEncodingException e) {
//                            e.printStackTrace();
//                        }
//                    }
                    paramMap.put(fieldName, value);
                } catch (IllegalAccessException e) {
                    e.printStackTrace();
                }
            }
        }
        StringBuilder stringBuilder = new StringBuilder();
        if (MapUtil.isEmpty(paramMap)) {
            return "";
        }
        for (Map.Entry<String, String> keyValueEntry : paramMap.entrySet()) {
            stringBuilder.append("&").append(keyValueEntry.getKey()).append("=").append(keyValueEntry.getValue());
        }
        return StringUtil.removeStart(stringBuilder.toString(), "&");
    }

    public Map<String, String> getQueryStringMap(String url) {
        Map<String, String> queryStringMap = new HashMap<>();
        int queryFlagIndex = url.indexOf(QUESTION_MARK);
        if (queryFlagIndex == -1) {
            return Collections.emptyMap();
        }
        String queryString = url.substring(queryFlagIndex + 1);
        String[] queryParams = queryString.split(AND);
        for (String param : queryParams) {
            if (StringUtil.isBlank(param)) {
                continue;
            }
            int paramIndex = param.indexOf(EQUAL);
            String key = param.substring(0, paramIndex);
            String value = param.substring(paramIndex + 1);
            queryStringMap.put(key, value);
        }
        return queryStringMap;
    }
}
