package cn.com.servyou.rmi.client;

import cn.com.servyou.ca.dto.CaConfigDTO;
import cn.com.servyou.ca.service.CaService;
import cn.com.servyou.ca.service.SignService;
import cn.com.servyou.ca.service.impl.CaServiceImpl;
import cn.com.servyou.ca.service.impl.SignServiceImpl;
import cn.com.servyou.constant.ServyouKeyConstant;
import cn.com.servyou.messagehandle.BaseMessageHandler;
import cn.com.servyou.messagehandle.MessageHandlerProvider;
import cn.com.servyou.messagehandle.MessageHanleEnum;
import cn.com.servyou.messagehandle.SecretConfiguration;
import cn.com.servyou.rmi.InvokeManage;
import cn.com.servyou.rmi.InvokeManageFactory;
import cn.com.servyou.rmi.ServerManage;
import cn.com.servyou.rmi.ServerManageFactory;
import cn.com.servyou.rmi.http.RequestMethod;
import cn.com.servyou.service.BackupDeclareDataRequest;
import cn.com.servyou.service.CommissionRequest;
import cn.com.servyou.service.CommonRequest;
import cn.com.servyou.service.CompanyRequest;
import cn.com.servyou.service.CustomerRequest;
import cn.com.servyou.service.DeclarationExportRequest;
import cn.com.servyou.service.DeclarationRequest;
import cn.com.servyou.service.DeferredTaxRequest;
import cn.com.servyou.service.DisputeRequest;
import cn.com.servyou.service.EITVerificationMessageRequest;
import cn.com.servyou.service.EmployeeInformationChangeRequest;
import cn.com.servyou.service.EmployeeRequest;
import cn.com.servyou.service.EquityIncentiveRequest;
import cn.com.servyou.service.EquitytTransferMessageRequest;
import cn.com.servyou.service.FileRequest;
import cn.com.servyou.service.IdentificationTaxRequest;
import cn.com.servyou.service.MergedArchivesRequest;
import cn.com.servyou.service.MessageRequest;
import cn.com.servyou.service.OutBusinessLicens;
import cn.com.servyou.service.PaymentRequest;
import cn.com.servyou.service.PersonalDeclarationRequest;
import cn.com.servyou.service.PersonalPensionRequest;
import cn.com.servyou.service.PreDeductDeductionAmountRequest;
import cn.com.servyou.service.SalaryCoreRequest;
import cn.com.servyou.service.SalaryImportRequest;
import cn.com.servyou.service.SpecialRequest;
import cn.com.servyou.service.StatisticRequest;
import cn.com.servyou.service.TaxPaymentRecordRequest;
import cn.com.servyou.service.TaxRequest;
import cn.com.servyou.service.TransferRequest;
import cn.com.servyou.service.socins.HrssCaRequest;
import cn.com.servyou.service.socins.HrssPreconditionFileRequest;
import cn.com.servyou.service.socins.HrssRequest;
import cn.com.servyou.service.socins.HrssShangHaiRequest;
import cn.com.servyou.service.socins.ImpositionRequest;
import cn.com.servyou.utils.DependUtil;
import cn.com.servyou.utils.LoggerUtil;
import cn.com.servyou.utils.MapUtil;
import cn.com.servyou.utils.ReadIniFileUtil;
import cn.com.servyou.utils.StringUtil;
import lombok.Data;
import lombok.extern.slf4j.Slf4j;

import java.io.IOException;
import java.lang.annotation.Annotation;
import java.lang.reflect.Proxy;
import java.util.HashMap;
import java.util.Map;
import java.util.Objects;
import java.util.UUID;

/**
 * 接口代理工厂
 *
 * @author zhouww
 * @since 20191024
 */
@Data
@Slf4j
public class ClientProxyFactory implements ServerManage, InvokeManage {

    /**
     * 环境对应一种配置
     */
    private String environmental;

    /**
     * appKey
     */
    private String appKey;

    /**
     * appSecret
     */
    private String appSecret;

    /**
     * ca相关配置
     */
    private CaConfigDTO caConfigDTO;

    /**
     * 日志开关是否开启，默认关闭
     */
    private Boolean logSwitchOpen;

    public CaConfigDTO getCaConfigDTO() {
        return caConfigDTO;
    }

    public void setCaConfigDTO(CaConfigDTO caConfigDTO) {
        if (caConfigDTO != null && StringUtil.isNotBlank(caConfigDTO.getCaUserConfigFile())) {
            try {
                Map<String, String> config = ReadIniFileUtil.readInfile(caConfigDTO.getCaUserConfigFile());
                if (MapUtil.isNotEmpty(config)) {
                    String logSwitch = config.get(ServyouKeyConstant.LOG_SWITCH);
                    if (Objects.equals(logSwitch, ServyouKeyConstant.ENABLE)) {
                        this.logSwitchOpen = Boolean.TRUE;
                    }
                }
            } catch (IOException e) {
                LoggerUtil.info(log, "获取日志开关异常，配置地址：{0}", caConfigDTO.getCaUserConfigFile());
            }
        }
        this.caConfigDTO = caConfigDTO;
    }

    /**
     * 服务管理器
     */
    private ServerManage serverManage;

    /**
     * 请求调用管理器
     */
    private InvokeManage invokeManage;

    /**
     * 存储生成的实例
     */
    private Map<String, Object> requestMap = new HashMap<>();

    /**
     * ca服务
     */
    private CaService caService;

    /**
     * 桌面加密机
     */
    private SignService signService;

    /**
     * 报文加密解密处理
     */
    private BaseMessageHandler baseMessageHandler;

    public ClientProxyFactory() {
        this.environmental = UUID.randomUUID().toString().replaceAll("-", "");
        serverManage = ServerManageFactory.create(environmental);
        invokeManage = InvokeManageFactory.create();
        this.logSwitchOpen = Boolean.FALSE;
        checkDepend();
    }

    public ClientProxyFactory(String appKey, String appSecret, String url) {
        this.environmental = UUID.randomUUID().toString().replaceAll("-", "");
        this.appKey = appKey;
        this.appSecret = appSecret;
        serverManage = ServerManageFactory.create(environmental, url);
        invokeManage = InvokeManageFactory.create();
        this.logSwitchOpen = Boolean.FALSE;
        checkDepend();
    }

    public ClientProxyFactory(String appKey, String appSecret, String url, SecretConfiguration configuration) {
        this.environmental = UUID.randomUUID().toString().replaceAll("-", "");
        this.appKey = appKey;
        this.appSecret = appSecret;
        serverManage = ServerManageFactory.create(environmental, url);
        invokeManage = InvokeManageFactory.create();
        this.logSwitchOpen = Boolean.FALSE;
        checkDepend();
        this.baseMessageHandler = MessageHandlerProvider.create(MessageHanleEnum.SM, configuration);
        this.baseMessageHandler.checkParam();
    }

    public ClientProxyFactory(String appKey, String appSecret, String environmental, String url) {
        this.environmental = environmental;
        this.appKey = appKey;
        this.appSecret = appSecret;
        serverManage = ServerManageFactory.create(environmental, url);
        invokeManage = InvokeManageFactory.create();
        this.logSwitchOpen = Boolean.FALSE;
        checkDepend();
    }

    /**
     * 检测依赖
     */
    public void checkDepend() {
        if (DependUtil.checkSlf4j() && DependUtil.checkJackson()) {
            return;
        }
        throw new RuntimeException("必须依赖slf4j和jackson");
    }

    public <T> T getRequestProxy(Class serviceClientClazz) {
        if ( null == serviceClientClazz) {
            return null;
        }
        Annotation annotation = serviceClientClazz.getAnnotation(ServiceClient.class);
        if (null == annotation) {
            return null;
        }

        ClientProxy clientProxy = new ClientProxy(serviceClientClazz, this);
        LoggerUtil.info(log, "创建接口代理类 serviceName={0}", serviceClientClazz.getName());
        return (T) Proxy.newProxyInstance(this.getClass().getClassLoader(), new Class[]{serviceClientClazz}, clientProxy);
    }

    @Override
    public String getHost(String hostname) {
        return serverManage.getHost(hostname);
    }

    public CaService getCAService() {
        if (null == caService) {
            synchronized (this) {
                if (null == caService) {
                    caService = new CaServiceImpl(this);
                }
            }
        }
        return caService;
    }

    public SignService getSignService() {
        if (null == signService) {
            synchronized (this) {
                if (null == signService) {
                    if (null != caConfigDTO && StringUtil.isNotBlank(caConfigDTO.getCaUserConfigFile())) {
                        signService = new SignServiceImpl(caConfigDTO.getCaUserConfigFile());
                    }
                }
            }
        }
        return signService;
    }

    @Override
    public void addServerConfig(String serverName, String host) {
        serverManage.addServerConfig(serverName, host);
    }

    @Override
    public byte[] request(RequestMethod method, String url, Map<String, String> headers, String body) {
        try {
            if (!Objects.isNull(this.baseMessageHandler) && !Objects.isNull(body)) {
                //入参进行加密
                body = this.baseMessageHandler.encrypt(body);
                LoggerUtil.info(log, "加密后的入参={0}", body);
            }
            if (MapUtil.isEmpty(headers)) {
                headers = new HashMap<>(2 << 2);
            }
            String traceId = UUID.randomUUID().toString().replace("-", "");
            headers.put(ServyouKeyConstant.TRACE_ID, traceId);
            LoggerUtil.info(log, "该{0}请求的traceId={1}", url, traceId);
            return invokeManage.request(method, url, headers, body);
        } catch (Exception e) {
            LoggerUtil.error(log, e, "接口请求发生错误");
            throw new RuntimeException(e);
        }
    }

    /**
     * 代扣代缴企业查询接口
     *
     * @return
     */
    public CompanyRequest getCompanyRequest() {
        return getCacheRequest(CompanyRequest.class);
    }

    /**
     * 代扣代缴专项查询接口
     *
     * @return
     */
    public SpecialRequest getSpecialQueryRequest() {
        return getCacheRequest(SpecialRequest.class);
    }

    /**
     * 代扣代缴申报接口
     *
     * @return
     */
    public EmployeeInformationChangeRequest getEmployeeInformationChange() {
        return getCacheRequest(EmployeeInformationChangeRequest.class);
    }

    public DeclarationRequest getDeclarationRequest() {
        return getCacheRequest(DeclarationRequest.class);
    }

    /**
     * 代扣代缴人员相关接口
     *
     * @return
     */
    public EmployeeRequest getEmployeeRequest() {
        return getCacheRequest(EmployeeRequest.class);
    }

    public PaymentRequest getPaymentRequest() {
        return getCacheRequest(PaymentRequest.class);
    }

    public TaxRequest getTaxRequest() {
        return getCacheRequest(TaxRequest.class);
    }

    public DeclarationExportRequest getDeclarationExportRequest() {
        return getCacheRequest(DeclarationExportRequest.class);
    }

    public SalaryImportRequest getSalaryImportRequest() {
        return getCacheRequest(SalaryImportRequest.class);
    }

    /**
     * 税费种认定情况
     *
     * @return
     */
    public IdentificationTaxRequest getIdentificationTaxRequest() {
        return getCacheRequest(IdentificationTaxRequest.class);
    }

    /**
     * 外经证查询接口
     *
     * @return
     */
    public OutBusinessLicens getOutBusinessLicens() {
        return getCacheRequest(OutBusinessLicens.class);
    }

    public SalaryCoreRequest getSalaryCoreRequest() {
        return getCacheRequest(SalaryCoreRequest.class);
    }

    public FileRequest getFileRequest() {
        return getCacheRequest(FileRequest.class);
    }

    /**
     * 通用请求
     *
     * @return
     */
    public CommonRequest getCommonRequest() {
        return getCacheRequest(CommonRequest.class);
    }

    /**
     * 申述接口
     *
     * @return
     */
    public DisputeRequest getDisputeRequest() {
        return getCacheRequest(DisputeRequest.class);
    }

    /**
     * 人员并档接口
     *
     * @return
     */
    public MergedArchivesRequest getMergedArchivesRequest() {
        return getCacheRequest(MergedArchivesRequest.class);
    }

    /**
     * 预扣预缴接口
     *
     * @return
     */
    public PreDeductDeductionAmountRequest getPreDeductDeductionAmountRequest() {
        return getCacheRequest(PreDeductDeductionAmountRequest.class);
    }

    /**
     * 人社接口
     *
     * @return
     */
    public HrssRequest getHrssRequest() {
        return getCacheRequest(HrssRequest.class);
    }

    public HrssCaRequest getHrssCaRequest() {
        return getCacheRequest(HrssCaRequest.class);
    }

    /**
     * 人社-上海地区特色接口
     *
     * @return
     */
    public HrssShangHaiRequest getHrssShangHaiRequest() {
        return getCacheRequest(HrssShangHaiRequest.class);
    }

    /**
     * 参停保材料下载接口
     *
     * @return
     */
    public HrssPreconditionFileRequest getHrssPreconditionFileRequest() {
        return getCacheRequest(HrssPreconditionFileRequest.class);
    }

    /**
     * 社保接口
     *
     * @return
     */
    public ImpositionRequest getImpositionRequest() {
        return getCacheRequest(ImpositionRequest.class);
    }

    /**
     * 获取获取流量统计
     *
     * @return
     */
    public StatisticRequest getStatisticRequest() {
        return getCacheRequest(StatisticRequest.class);
    }

    /**
     * 股权激励相关接口
     *
     * @return
     */
    public EquityIncentiveRequest getEquityIncentiveRequest() {
        return getCacheRequest(EquityIncentiveRequest.class);
    }

    /**
     * 延纳税报送相关接口
     *
     * @return
     */
    public DeferredTaxRequest getDeferredTaxRequest() {
        return getCacheRequest(DeferredTaxRequest.class);
    }

    /**
     * 缴税记录查询
     *
     * @return
     */
    public TaxPaymentRecordRequest queryTaxPayment() {
        return getCacheRequest(TaxPaymentRecordRequest.class);
    }

    /**
     * 消息查询
     *
     * @return
     */
    public MessageRequest getMessageRequest() {
        return getCacheRequest(MessageRequest.class);
    }

    /**
     * 股权转让信息核对反馈
     *
     * @return
     */
    public EquitytTransferMessageRequest getEquitytTransferMessageRequest() {
        return getCacheRequest(EquitytTransferMessageRequest.class);
    }

    /**
     * 企业所得税核查
     *
     * @return
     */
    public EITVerificationMessageRequest getEITVerificationMessageRequest() {
        return getCacheRequest(EITVerificationMessageRequest.class);
    }

    /**
     * 查询手续费退税信息
     *
     * @return
     */
    public CommissionRequest getCommissionRequest() {
        return getCacheRequest(CommissionRequest.class);
    }

    /**
     * 个人的申报情况查询
     *
     * @return
     */
    public PersonalDeclarationRequest queryPersonalDeclarationDetail() {
        return getCacheRequest(PersonalDeclarationRequest.class);
    }

    /**
     * 下载个人养老金
     *
     * @return
     */
    public PersonalPensionRequest getPersonalDeclarationRequest() {
        return getCacheRequest(PersonalPensionRequest.class);
    }

    public CustomerRequest getCustomerRequest() {
        return getCacheRequest(CustomerRequest.class);
    }

    /**
     * its数据迁移
     *
     * @return
     */
    public TransferRequest getTransferRequest() {
        return getCacheRequest(TransferRequest.class);
    }

    public BackupDeclareDataRequest getBackupDeclareDataRequest() {
        return getCacheRequest(BackupDeclareDataRequest.class);
    }

    public <T> T getCacheRequest(Class requestClazz) {
        String fullPath = requestClazz.getName();
        if (null == requestMap.get(fullPath)) {
            synchronized (requestClazz) {
                if (null == requestMap.get(fullPath)) {
                    requestMap.put(fullPath, this.getRequestProxy(requestClazz));
                }
            }
        }
        return (T) requestMap.get(fullPath);
    }
}