package cn.com.servyou.service.scene;

import cn.com.servyou.dto.enums.DataCheckEnum;
import cn.com.servyou.dto.enums.DeclarationAppendixItemEnum;
import cn.com.servyou.dto.enums.DeclarationTypeEnum;
import cn.com.servyou.dto.enums.EmployeeLicenseTypeEnum;
import cn.com.servyou.dto.enums.IncomeItemEnum;
import cn.com.servyou.dto.tax.CalculateIndividualIncomeTaxRequest;
import cn.com.servyou.dto.tax.ClassificationIncome;
import cn.com.servyou.dto.tax.ClassificationIncomeRequest;
import cn.com.servyou.dto.tax.ComplexIncome;
import cn.com.servyou.dto.tax.ComplexIncomeRequest;
import cn.com.servyou.dto.tax.ExemptIncome;
import cn.com.servyou.dto.tax.ExtensionEndowmentInsurance;
import cn.com.servyou.dto.tax.IncomeDeduction;
import cn.com.servyou.dto.tax.Insurance;
import cn.com.servyou.dto.tax.InsuranceMarketingIncome;
import cn.com.servyou.dto.tax.NonResidentsIncome;
import cn.com.servyou.dto.tax.NonResidentsIncomeRequest;
import cn.com.servyou.dto.tax.NormalSalarySpecIncome;
import cn.com.servyou.dto.tax.SecuritiesBrokerIncome;
import cn.com.servyou.utils.CollectionUtil;
import cn.com.servyou.utils.IDCardUtil;
import cn.com.servyou.utils.LoggerUtil;
import cn.com.servyou.utils.StringUtil;
import lombok.extern.slf4j.Slf4j;

import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * 数据校验
 * 校验数据的合规性
 *
 * @author zhouww
 * @since 20191024
 */
@Slf4j
public class LicenseCheck {
    private LicenseCheck() {
        // 校验类，禁止实例化
    }

    /**
     * 校验工资薪金和附表
     * 校验证照类型和证照号码不能为空
     * 校验证照类型
     * 校验居民身份证号码
     * @return 校验结果
     */
    public static List<DataCheckResult> checkIncome(CalculateIndividualIncomeTaxRequest request) {
        List<DataCheckResult> checkResultList = new ArrayList<>();
        List<DataCheckResult> complexIncomeRequestList = checkComplexIncomeRequest(request.getComplexIncomeRequest());
        List<DataCheckResult> classificationIncomeRequestList = checkClassificationIncomeRequest(request.getClassificationIncomeRequest());
        List<DataCheckResult> nonResidentsIncomeRequestList = checkNonResidentsIncomeRequest(request.getNonResidentsIncomeRequest());
        checkResultList.addAll(complexIncomeRequestList);
        checkResultList.addAll(classificationIncomeRequestList);
        checkResultList.addAll(nonResidentsIncomeRequestList);
        return checkResultList;
    }

    /**
     * 校验综合所得的薪资
     *
     * @param complexIncomeRequest complexIncomeRequest
     * @return 校验结果
     */
    public static List<DataCheckResult> checkComplexIncomeRequest(ComplexIncomeRequest complexIncomeRequest) {
        if (null == complexIncomeRequest) {
            return Collections.emptyList();
        }
        List<DataCheckResult> complexIncomeCheckList = new ArrayList<>();
        NormalSalarySpecIncome normalSalarySpecIncome = complexIncomeRequest.getNormalSalarySpecIncome();
        if (null != normalSalarySpecIncome) {
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(normalSalarySpecIncome.getNormalSalarySpec(), IncomeItemEnum.NORMAL_SALARIES));
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(normalSalarySpecIncome.getLastMonthNormalSalary(), IncomeItemEnum.NORMAL_SALARIES));
        }
        InsuranceMarketingIncome insuranceMarketingIncome = complexIncomeRequest.getInsuranceMarketingIncome();
        if (null != insuranceMarketingIncome) {
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(insuranceMarketingIncome.getInsuranceMarketingIncomeList(), IncomeItemEnum.INSURANCE_SALESMEN));
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(insuranceMarketingIncome.getLastMonthSalary(), IncomeItemEnum.INSURANCE_SALESMEN));
        }
        SecuritiesBrokerIncome securitiesBrokerIncome = complexIncomeRequest.getSecuritiesBrokerIncome();
        if (null != securitiesBrokerIncome) {
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(securitiesBrokerIncome.getSecuritiesBrokerIncomeList(), IncomeItemEnum.SECURITIES_BROKER));
            complexIncomeCheckList.addAll(checkComplexIncomeProperty(securitiesBrokerIncome.getLastMonthSalary(), IncomeItemEnum.SECURITIES_BROKER));
        }
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getStockIncentiveList(), IncomeItemEnum.EQUITY_INCENTIVE));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getRoyaltyFeeList(), IncomeItemEnum.CONCESSION));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getRetreatIncome(), IncomeItemEnum.RETREAT));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getCompensateIncome(), IncomeItemEnum.COMPENSATE));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getLaborRemunerationIncome(), IncomeItemEnum.LABOR));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getAuthorsRemunerationIncome(), IncomeItemEnum.ROYALTIES));

        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getEnterpriseAnnuityList(), IncomeItemEnum.ANNUITY_RECEIPT));
        complexIncomeCheckList.addAll(checkComplexIncomeProperty(complexIncomeRequest.getAnnualOneTimeBonusIncome(), IncomeItemEnum.BONUS));

        // 附表校验
        complexIncomeCheckList.addAll(checkInsurance(complexIncomeRequest.getInsuranceList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        complexIncomeCheckList.addAll(checkIncomeDeduction(complexIncomeRequest.getIncomeDeductionList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        complexIncomeCheckList.addAll(checkExtensionEndowmentInsurance(complexIncomeRequest.getExtensionEndowmentInsuranceList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME ));
        complexIncomeCheckList.addAll(checkExemptIncome(complexIncomeRequest.getExemptIncomeList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        return complexIncomeCheckList;
    }

    /**
     * 校验分类的薪金
     * @param classificationIncomeRequest classificationIncomeRequest
     * @return 校验结果
     */
    public static List<DataCheckResult>  checkClassificationIncomeRequest(ClassificationIncomeRequest classificationIncomeRequest) {
        if (null == classificationIncomeRequest) {
            return Collections.emptyList();
        }
        List<DataCheckResult> complexIncomeCheckList = new ArrayList<>();
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getInterestDividendBonusIncomes(), IncomeItemEnum.INTEREST));
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getEquitytTransferIncomes(), IncomeItemEnum.EQUITY));
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getOtherPropertyAssignmentIncomes(), IncomeItemEnum.PROPERTY_TRANSFER));
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getOtherPropertyRentalIncomes(), IncomeItemEnum.OTHER_PROPERTY_RENTAL));
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getAccidentalIncome(), IncomeItemEnum.HAPHAZARD));
        complexIncomeCheckList.addAll(checkClassificationIncomeProperty(classificationIncomeRequest.getListedCompanyInterest(), IncomeItemEnum.LISTED_COMPANY_INTEREST));

        // 附表校验
        complexIncomeCheckList.addAll(checkIncomeDeduction(classificationIncomeRequest.getIncomeDeductionList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        complexIncomeCheckList.addAll(checkExemptIncome(classificationIncomeRequest.getExemptIncomeList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        return complexIncomeCheckList;
    }

    /**
     * 校验非居民的薪金
     * @param nonResidentsIncomeRequest nonResidentsIncomeRequest
     * @return 校验结果
     */
    public static List<DataCheckResult> checkNonResidentsIncomeRequest(NonResidentsIncomeRequest nonResidentsIncomeRequest) {
        if (null == nonResidentsIncomeRequest) {
            return Collections.emptyList();
        }
        List<DataCheckResult> complexIncomeCheckList = new ArrayList<>();
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getForeignerWagesSalaryIncomes(), IncomeItemEnum.ABROAD));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getForeignerMonthsBonusIncomes(), IncomeItemEnum.MONTHS_BONUS));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getLaborRemunerationIncomes(), IncomeItemEnum.LABOR));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getAuthorsRemunerationIncome(), IncomeItemEnum.ROYALTIES));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getInterestDividendBonusIncomes(), IncomeItemEnum.INTEREST));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getEquitytTransferIncomes(), IncomeItemEnum.EQUITY));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getOtherPropertyAssignmentIncomes(), IncomeItemEnum.PROPERTY_TRANSFER));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getAccidentalIncome(), IncomeItemEnum.HAPHAZARD));
        complexIncomeCheckList.addAll(checkNonResidentsIncomeProperty(nonResidentsIncomeRequest.getCompensateIncomes(), IncomeItemEnum.COMPENSATE));

        // 附表校验
        complexIncomeCheckList.addAll(checkIncomeDeduction(nonResidentsIncomeRequest.getIncomeDeductionList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        complexIncomeCheckList.addAll(checkExemptIncome(nonResidentsIncomeRequest.getExemptIncomeList(), DeclarationTypeEnum.NON_RESIDENTS_INCOME));
        return complexIncomeCheckList;
    }

    /**
     * 校验综合所得的薪资数据
     *
     * @param complexIncomeList complexIncomeList
     * @param itemCode itemCode
     * @return 校验结果
     */
    public static List<DataCheckResult> checkComplexIncomeProperty(List<ComplexIncome> complexIncomeList, IncomeItemEnum itemCode) {
        if (CollectionUtil.isEmpty(complexIncomeList)) {
            return Collections.emptyList();
        }
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (ComplexIncome complexIncome : complexIncomeList) {
            DataCheckResult dataCheckResult = checkLicense(complexIncome.getLicenseType(), complexIncome.getLicenseNumber(),
                    DeclarationTypeEnum.COMPLEX_INCOME, itemCode, null);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 校验分类所得的薪资数据
     * @param classificationIncomeList classificationIncomeList
     * @param itemCode itemCode
     * @return 校验结果
     */
    public static List<DataCheckResult> checkClassificationIncomeProperty(List<ClassificationIncome> classificationIncomeList, IncomeItemEnum itemCode) {
        if (CollectionUtil.isEmpty(classificationIncomeList)) {
            return Collections.emptyList();
        }
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (ClassificationIncome classificationIncome : classificationIncomeList) {
            DataCheckResult dataCheckResult = checkLicense(classificationIncome.getLicenseType(), classificationIncome.getLicenseNumber(),
                    DeclarationTypeEnum.CLASSIFICATION_INCOME, itemCode, null);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 校验非居民的薪资数据
     *
     * @param nonResidentsIncomeList nonResidentsIncomeList
     * @param itemCode itemCode
     * @return 校验结果
     */
    public static List<DataCheckResult> checkNonResidentsIncomeProperty(List<NonResidentsIncome> nonResidentsIncomeList, IncomeItemEnum itemCode) {
        if (CollectionUtil.isEmpty(nonResidentsIncomeList)) {
            return Collections.emptyList();
        }
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (NonResidentsIncome nonResidentsIncome : nonResidentsIncomeList) {
            DataCheckResult dataCheckResult = checkLicense(nonResidentsIncome.getLicenseType(), nonResidentsIncome.getLicenseNumber(),
                    DeclarationTypeEnum.NON_RESIDENTS_INCOME, itemCode, null);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 商业健康险附表
     * @param insuranceList insuranceList
     * @param declarationTypeEnum declarationTypeEnum
     * @return 校验结果
     */
    public static List<DataCheckResult> checkInsurance(List<Insurance> insuranceList, DeclarationTypeEnum declarationTypeEnum) {
        if (CollectionUtil.isEmpty(insuranceList)) {
            return Collections.emptyList();
        }
        LoggerUtil.info(log, "开始校验商业健康险附表");
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (Insurance insurance : insuranceList) {
            DataCheckResult dataCheckResult = checkLicense(insurance.getLicenseType(), insurance.getLicenseNumber(),
                    declarationTypeEnum, null, DeclarationAppendixItemEnum.INSURANCE);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 减免附表
     * @param incomeDeductionList  incomeDeductionList
     * @param declarationTypeEnum declarationTypeEnum
     * @return 校验结果
     */
    public static List<DataCheckResult> checkIncomeDeduction(List<IncomeDeduction> incomeDeductionList, DeclarationTypeEnum declarationTypeEnum) {
        if (CollectionUtil.isEmpty(incomeDeductionList)) {
            return Collections.emptyList();
        }
        LoggerUtil.info(log, "开始校验减免附表");
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (IncomeDeduction incomeDeduction : incomeDeductionList) {
            DataCheckResult dataCheckResult = checkLicense(incomeDeduction.getLicenseType(), incomeDeduction.getLicenseNumber(),
                    declarationTypeEnum, null, DeclarationAppendixItemEnum.INCOME_DEDUCTION );
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 税延养老保险附表
     * @param extensionEndowmentInsuranceList extensionEndowmentInsuranceList
     * @param declarationTypeEnum declarationTypeEnum
     * @return 校验结果
     */
    public static List<DataCheckResult> checkExtensionEndowmentInsurance(List<ExtensionEndowmentInsurance> extensionEndowmentInsuranceList, DeclarationTypeEnum declarationTypeEnum) {
        if (CollectionUtil.isEmpty(extensionEndowmentInsuranceList)) {
            return Collections.emptyList();
        }
        LoggerUtil.info(log, "开始校验税延养老保险附表");
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (ExtensionEndowmentInsurance extensionEndowmentInsurance : extensionEndowmentInsuranceList) {
            DataCheckResult dataCheckResult = checkLicense( extensionEndowmentInsurance.getLicenseType(), extensionEndowmentInsurance.getLicenseNumber(),
                    declarationTypeEnum, null , DeclarationAppendixItemEnum.EXTENSION_ENDOWMENT_INSURANCE);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 免税附表
     * @param exemptIncomeList exemptIncomeList
     * @param declarationTypeEnum declarationTypeEnum
     * @return 校验结果
     */
    public static List<DataCheckResult> checkExemptIncome(List<ExemptIncome> exemptIncomeList, DeclarationTypeEnum declarationTypeEnum) {
        if (CollectionUtil.isEmpty(exemptIncomeList)) {
            return Collections.emptyList();
        }
        LoggerUtil.info(log, "开始校验免税附表");
        List<DataCheckResult> dataCheckResultList = new ArrayList<>();
        for (ExemptIncome exemptIncome : exemptIncomeList) {
            DataCheckResult dataCheckResult = checkLicense(exemptIncome.getLicenseType(), exemptIncome.getLicenseNumber(), declarationTypeEnum, null , DeclarationAppendixItemEnum.EXEMPT_INCOME);
            if (null != dataCheckResult) {
                dataCheckResultList.add(dataCheckResult);
            }
        }
        return dataCheckResultList;
    }

    /**
     * 数据校验
     * @param licenseType licenseType
     * @param licenseNumber licenseNumber
     * @param declarationTypeEnum declarationTypeEnum
     * @param itemCode itemCode
     * @param tableName 报表名字
     * @return 校验结果
     */
    private static DataCheckResult checkLicense(String licenseType, String licenseNumber, DeclarationTypeEnum declarationTypeEnum, IncomeItemEnum itemCode, DeclarationAppendixItemEnum tableName){
        // 校验照类型和证照号码 是否为空
        if (StringUtil.isAllBlank(licenseNumber, licenseNumber)) {
            return createCheckResult(licenseType, licenseNumber, declarationTypeEnum, itemCode, DataCheckEnum.EMPLOYEE_LICENSE_EMPTY.getCode(),
                    DataCheckEnum.EMPLOYEE_LICENSE_EMPTY.getDesc(), tableName);
        } else if (StringUtil.isBlank(licenseType)) {
            return createCheckResult(licenseType, licenseNumber, declarationTypeEnum, itemCode, DataCheckEnum.EMPLOYEE_LICENSE_TYPE_EMPTY.getCode(),
                    DataCheckEnum.EMPLOYEE_LICENSE_TYPE_EMPTY.getDesc(), tableName);
        } else if (StringUtil.isBlank(licenseNumber)) {
            return createCheckResult(licenseType, licenseNumber, declarationTypeEnum, itemCode, DataCheckEnum.EMPLOYEE_LICENSE_NUMBER_EMPTY.getCode(),
                    DataCheckEnum.EMPLOYEE_LICENSE_NUMBER_EMPTY.getDesc(), tableName);
        }

        // 经过上面校验，证照类型和证照号码不能为空，接着校验证照类型和证照号码
        EmployeeLicenseTypeEnum employeeLicenseTypeEnum = EmployeeLicenseTypeEnum.getByShortDesc(licenseType);
        if (employeeLicenseTypeEnum == null) {
            return createCheckResult(licenseType, licenseNumber, declarationTypeEnum, itemCode, DataCheckEnum.EMPLOYEE_LICENSE_TYPE_ERROR.getCode(),
                    DataCheckEnum.EMPLOYEE_LICENSE_TYPE_ERROR.getDesc(), tableName);
        } else if (employeeLicenseTypeEnum != null && employeeLicenseTypeEnum == EmployeeLicenseTypeEnum.RESIDENT_IDENTITY_CARD) {
            // 只需要校验居民身份证
            String errorMessage = IDCardUtil.verify(licenseNumber);
            if (StringUtil.isNotBlank(errorMessage)) {
                return createCheckResult(licenseType, licenseNumber, declarationTypeEnum, itemCode, DataCheckEnum.EMPLOYEE_LICENSE_NUMBER_ERROR.getCode(),
                        errorMessage, tableName);
            }
        }
        return null;
    }

    private static DataCheckResult createCheckResult(String licenseType,
                                                     String licenseNumber,
                                                     DeclarationTypeEnum declarationTypeEnum,
                                                     IncomeItemEnum itemCode,
                                                     String messageCode,
                                                     String errorMessage,
                                                     DeclarationAppendixItemEnum tableName) {
        DataCheckResult dataCheckResult = new DataCheckResult();
        dataCheckResult.setDeclarationTypeEnum(declarationTypeEnum);
        dataCheckResult.setIncomeItemEnum(itemCode);
        dataCheckResult.setMessageCode(messageCode);
        dataCheckResult.setErrorMessage(errorMessage);
        dataCheckResult.setLicenseType(licenseType);
        dataCheckResult.setLicenseNumber(licenseNumber);
        dataCheckResult.setTableName(tableName);
        return dataCheckResult;
    }
}
