package cn.com.servyou.service.scene;

import cn.com.servyou.dto.PersonUniqueKey;
import cn.com.servyou.dto.declare.CompanyDeclareRequest;
import cn.com.servyou.dto.tax.CalculateIndividualIncomeTaxRequest;
import cn.com.servyou.dto.tax.ComplexIncome;
import cn.com.servyou.dto.tax.ComplexIncomeRequest;
import cn.com.servyou.dto.tax.DetailCompanyEmployee;
import cn.com.servyou.dto.tax.InsuranceMarketingIncome;
import cn.com.servyou.dto.tax.NormalSalarySpecIncome;
import cn.com.servyou.dto.tax.SecuritiesBrokerIncome;
import cn.com.servyou.utils.CollectionUtil;
import org.springframework.util.CollectionUtils;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

/**
 * 校验人员是否重复
 *
 * @author zhouww
 * @since 20191024
 */
public class RepeatEmployeeCheck {
    private RepeatEmployeeCheck(){
        // 校验类，禁止实例
    }

    /**
     * 校验申报/数据导入的人员
     * @param request  请求报文
     * @return 校验结果
     */
    public static List<DetailCompanyEmployee> checkDeclareData(CompanyDeclareRequest request) {
        if (null == request) {
            return Collections.emptyList();
        }

        List<DetailCompanyEmployee> employeeList = request.getCompanyEmployeeList();
        if (CollectionUtils.isEmpty(employeeList)) {
            // 没有人员列表
            throw new RuntimeException("申报数据没有人员信息");
        }
        return checkRepeatEmployee(employeeList);
    }

    /**
     * 校验算税人员
     *
     * @param request 请求报文
     * @return 校验结果
     */
    public static List<DetailCompanyEmployee> checkCalculateData(CalculateIndividualIncomeTaxRequest request){
        if (null == request || null == request.getComplexIncomeRequest()) {
            return Collections.emptyList();
        }
        List<DetailCompanyEmployee> employeeList = new ArrayList<>();

        ComplexIncomeRequest complexIncomeRequest = request.getComplexIncomeRequest();
        // 校验正常工资薪金
        NormalSalarySpecIncome normalSalarySpecIncome = complexIncomeRequest.getNormalSalarySpecIncome();
        employeeList.addAll(checkNormalSalarySpec(normalSalarySpecIncome));
        // 校验保险营销员
        InsuranceMarketingIncome insuranceMarketingIncome = complexIncomeRequest.getInsuranceMarketingIncome();
        employeeList.addAll(checkInsuranceMarketing(insuranceMarketingIncome));
        // 校验证券经纪人
        SecuritiesBrokerIncome securitiesBrokerIncome = complexIncomeRequest.getSecuritiesBrokerIncome();
        employeeList.addAll(checkSecuritiesBroker(securitiesBrokerIncome));

        return employeeList;
    }

    /**
     * 校验正常工资薪金的人员
     *
     * @param normalSalarySpecIncome normalSalarySpecIncome
     * @return 校验结果
     */
    public static List<DetailCompanyEmployee> checkNormalSalarySpec(NormalSalarySpecIncome normalSalarySpecIncome) {
        if (null == normalSalarySpecIncome) {
            return Collections.emptyList();
        }
        List<ComplexIncome> normalSalarySpec = normalSalarySpecIncome.getNormalSalarySpec();
        List<DetailCompanyEmployee> employeeList = normalSalarySpecIncome.getCompanyEmployeeList();
        if (CollectionUtil.isNotEmpty(normalSalarySpec) && CollectionUtil.isEmpty(employeeList)) {
            throw new RuntimeException("正常工资薪金没有人员列表");
        }

        return checkRepeatEmployee(employeeList);
    }

    /**
     * 校验保险营销员
     *
     * @param insuranceMarketingIncome insuranceMarketingIncome
     * @return 校验结果
     */
    public static List<DetailCompanyEmployee>  checkInsuranceMarketing(InsuranceMarketingIncome insuranceMarketingIncome) {
        if (null == insuranceMarketingIncome) {
            return Collections.emptyList();
        }
        List<ComplexIncome> insuranceMarketingIncomeList = insuranceMarketingIncome.getInsuranceMarketingIncomeList();
        List<DetailCompanyEmployee> employeeList = insuranceMarketingIncome.getCompanyEmployeeList();
        if (CollectionUtil.isNotEmpty(insuranceMarketingIncomeList) && CollectionUtil.isEmpty(employeeList)) {
            throw new RuntimeException("保险营销员没有人员列表");
        }

        return checkRepeatEmployee(employeeList);
    }

    /**
     * 校验证券经纪人
     *
     * @param securitiesBrokerIncome securitiesBrokerIncome
     * @return 校验结果
     */
    public static List<DetailCompanyEmployee>  checkSecuritiesBroker(SecuritiesBrokerIncome securitiesBrokerIncome) {
        if (null == securitiesBrokerIncome) {
            return Collections.emptyList();
        }
        List<ComplexIncome> securitiesBrokerIncomeList = securitiesBrokerIncome.getSecuritiesBrokerIncomeList();
        List<DetailCompanyEmployee> employeeList = securitiesBrokerIncome.getCompanyEmployeeList();
        if (CollectionUtil.isNotEmpty(securitiesBrokerIncomeList) && CollectionUtil.isEmpty(employeeList)) {
            throw new RuntimeException("证券经纪人没有人员列表");
        }

        return checkRepeatEmployee(employeeList);
    }

    /**
     * 校验人员
     *
     * @param employeeList employeeList
     * @return 校验结果
     */
    public static  List<DetailCompanyEmployee> checkRepeatEmployee(List<DetailCompanyEmployee> employeeList) {
        List<DetailCompanyEmployee> repeatEmployeeList = new ArrayList<>();
        Map<PersonUniqueKey, DetailCompanyEmployee> employeeMap = new HashMap();
        for (DetailCompanyEmployee detailCompanyEmployee : employeeList) {
            PersonUniqueKey personUniqueKey = new PersonUniqueKey(detailCompanyEmployee.getLicenseType(), detailCompanyEmployee.getLicenseNumber());
            DetailCompanyEmployee alreadyExists = employeeMap.get(personUniqueKey);
            if (null == alreadyExists) {
                employeeMap.put(personUniqueKey, detailCompanyEmployee);
            } else {
                repeatEmployeeList.add(detailCompanyEmployee);
                repeatEmployeeList.remove(alreadyExists);   // 解决大于3个的重复数据的场景
                repeatEmployeeList.add(alreadyExists);
            }
        }
        return repeatEmployeeList;
    }
}
