package cn.com.servyou.utils;

import lombok.extern.slf4j.Slf4j;

import javax.crypto.Cipher;
import javax.crypto.Mac;
import javax.crypto.spec.SecretKeySpec;
import java.io.UnsupportedEncodingException;
import java.net.URLEncoder;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.NoSuchAlgorithmException;
import java.security.SignatureException;
import java.util.Base64;
import java.util.HashMap;
import java.util.Map;
import java.util.TreeMap;

import static cn.com.servyou.constant.ServyouKeyConstant.APP_KEY;
import static cn.com.servyou.constant.ServyouKeyConstant.APP_SECRET;
import static cn.com.servyou.constant.ServyouKeyConstant.ENCODING_UTF8;
import static cn.com.servyou.constant.ServyouKeyConstant.H_MAC_SHA_256;
import static cn.com.servyou.constant.ServyouKeyConstant.REQUEST_ID;
import static cn.com.servyou.constant.ServyouKeyConstant.TIMESTAMP;
import static cn.com.servyou.constant.ServyouKeyConstant.VERSION;
import static cn.com.servyou.constant.ServyouKeyConstant.X_REQUEST_NONCE;

/**
 * 签名工具
 *
 * @author zhouww
 * @since 20190829
 */
@Slf4j
public class EncryptionUtil {

    private EncryptionUtil(){
        // 静态工具类 禁止实例化
    }

    /**
     * 参数加密
     * @param appKey  appKey
     * @param appSecret appSecret
     * @param timestamp timestamp
     * @param version version
     * @param xReqNonce xReqNonce
     * @param requestId 请求ID
     * @return 加密结果
     */
    public static String parameterEncryption(String appKey, String appSecret, String timestamp, String version, String xReqNonce, String requestId) throws SignatureException {
        Map<String, String> queryMap = new HashMap<>();
        queryMap.put(REQUEST_ID, requestId);

        return parameterEncryption(appKey, appSecret, timestamp, version, xReqNonce, queryMap);
    }

    /**
     * 参数加密
     * @param appKey  appKey
     * @param appSecret appSecret
     * @param timestamp timestamp
     * @param version version
     * @param xReqNonce xReqNonce
     * @param queryMap 请求行key-value
     * @return 加密结果
     */
    public static String parameterEncryption(String appKey, String appSecret, String timestamp, String version, String xReqNonce, Map<String, String> queryMap) throws SignatureException {
        // 将所有请求参数包括公共参数头key=value字段对按key字典序进行排序
        TreeMap<String, String> parameterMap = new TreeMap();
        parameterMap.put(VERSION, version);
        parameterMap.put(TIMESTAMP, timestamp);
        parameterMap.put(APP_KEY, appKey);
        parameterMap.put(X_REQUEST_NONCE, xReqNonce);
        parameterMap.put(APP_SECRET, appSecret);
        if (MapUtil.isNotEmpty(queryMap)) {
            for (Map.Entry<String, String> queryEntry : queryMap.entrySet()) {
                parameterMap.put(queryEntry.getKey(), queryEntry.getValue());
            }
        }
        return parameterEncryption(parameterMap);
    }

    /**
     * 参数加密
     * 必须有
     * @param treeMap 参数
     * @return 加密结果
     */
    public static String parameterEncryption(TreeMap<String, String> treeMap) throws SignatureException {
        LoggerUtil.info(log, "签名参数 params={0}", JSONUtil.toJSON(treeMap));
        //  将以上key=value对的value进行合并,生成一下字符串mergeStr
        StringBuilder mergeStr = new StringBuilder();
        for (Map.Entry<String, String> stringStringEntry : treeMap.entrySet()) {
            mergeStr.append(stringStringEntry.getValue());
        }
        LoggerUtil.info(log, "Key-Value合并： value={0}", mergeStr);

        // 将生成的mergeStr进行Url编码
        String encodedStr = null;
        try {
            encodedStr = URLEncoder.encode(mergeStr.toString(), ENCODING_UTF8);
        } catch (UnsupportedEncodingException e) {
            LoggerUtil.error(log, e, "URL编码失败");
            throw new RuntimeException("URL编码失败");
        }
        LoggerUtil.info(log, "URL编码 url={0}", encodedStr);

        // 利用HmacSHA256算法对signStr进行哈希运算生成消息摘要,摘要结果以Base64结果形式返回，signStr即为请求参数中的signature字段
        try {
            Mac mac = Mac.getInstance(H_MAC_SHA_256);
            String appSecret = treeMap.get(APP_SECRET);
            SecretKeySpec signingKey = new SecretKeySpec(appSecret.getBytes(), H_MAC_SHA_256);
            mac.init(signingKey);
            byte[] signData = mac.doFinal(encodedStr.getBytes());
            byte[] resultBytes = Base64.getEncoder().encode(signData);
            String signatureResult = new String(resultBytes, ENCODING_UTF8);
            LoggerUtil.info(log, "加密结果 signature={0}", signatureResult);
            return signatureResult;
        } catch (NoSuchAlgorithmException e) {
            LoggerUtil.error(log, e,"平台不支持 HmacSHA 摘要方式");
            throw new SignatureException("平台不支持 HmacSHA 摘要方式");
        } catch (InvalidKeyException e) {
            LoggerUtil.error(log, e,"Speicified access secret is not valid.");
            throw new SignatureException("Speicified access secret is not valid.");
        } catch (UnsupportedEncodingException e) {
            LoggerUtil.error(log, e, "加密结果转码失败");
            throw new RuntimeException("加密结果转码失败");
        }
    }

    /**
     * 密码加密
     * 实名密码和申报密码
     * @param appSecret appSecret
     * @param text text
     * @return
     */
    public static String encryption(String appSecret, String text) {
        try {
            //对明文密码进行加密
            byte[] appSecrets = Base64.getDecoder().decode(appSecret.getBytes());
            byte[] encryptedPasswords = SymmetricCryptoUtil.symmetricCrypto(text.getBytes(), appSecrets, "AES", Cipher.ENCRYPT_MODE);
            return new String(Base64.getEncoder().encode(encryptedPasswords), ENCODING_UTF8);
        } catch (GeneralSecurityException | UnsupportedEncodingException e) {
            LoggerUtil.error(log, e, "加密失败");
        }
        throw new RuntimeException("加密失败");
    }


    /**
     * 密码解密
     * @param appSecret appSecret
     * @param declarePassword declarePassword
     * @return
     */
    public static String decrypt(String appSecret, String declarePassword) {

        try {
            byte[] appSecrets = Base64.getDecoder().decode(appSecret.getBytes());
            byte[] declarePasswords = Base64.getDecoder().decode(declarePassword.getBytes());

            //解密后的密码
            byte[] decryptPasswords = SymmetricCryptoUtil.symmetricCrypto(declarePasswords,
                    appSecrets, "AES", Cipher.DECRYPT_MODE);
            return new String(decryptPasswords);
        } catch (GeneralSecurityException e) {
            e.printStackTrace();
        }
        throw new RuntimeException("解密失败");
    }

}
