package cn.com.servyou.utils;

import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.DeserializationFeature;
import com.fasterxml.jackson.databind.JavaType;
import com.fasterxml.jackson.databind.JsonNode;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.node.ArrayNode;
import com.fasterxml.jackson.databind.node.NullNode;
import com.fasterxml.jackson.databind.node.ObjectNode;

import java.io.IOException;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.function.Consumer;

/**
 * json序列化和反序列化
 *
 * @author zhouww
 * @since 20191024
 */
public class JSONUtil {
    private static final ObjectMapper objectMapper = new ObjectMapper();
    private static final ObjectMapper objectMapperExcludeNull = new ObjectMapper();

    static {
        objectMapper.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapperExcludeNull.configure(DeserializationFeature.FAIL_ON_UNKNOWN_PROPERTIES, false);
        objectMapperExcludeNull.setSerializationInclusion(JsonInclude.Include.NON_NULL);
    }

    /**
     * 序列化
     *
     * @param obj 需要序列化的对象
     * @return json字符串
     */
    public static String toJSON(Object obj) {
        try {
            return objectMapper.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 序列化
     * @param obj
     * @return
     */
    public static String toJSONExcludeNull(Object obj) {
        try {
            return objectMapperExcludeNull.writeValueAsString(obj);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 解析JSON
     *
     * @param json json 字符串
     * @param clazz 主类
     * @param <T> 返回的数据类型
     * @return 反序列化后的对象
     */
    public static <T> T parseJSON(String json, Class clazz) {
        try {
            return (T) objectMapper.readValue(json, clazz);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 范型反序列化
     *
     * @param json json字符串
     * @param clazz 主类
     * @param generalizationClasss 范型类型
     * @param <T> 返回类型
     * @return 反序列化后的对象
     */
    public static <T> T parseJSON(String json, Class clazz, Class... generalizationClasss) {

        JavaType javaType = objectMapper.getTypeFactory().constructParametricType(clazz, generalizationClasss);
        try {
            return objectMapper.readValue(json, javaType);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    public static <T> T parseJSON(String json, JavaType javaType) {
        try {
            return objectMapper.readValue(json, javaType);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * json 增加 属性
     *
     * @param json  字符串
     * @param key   key
     * @param value 值
     * @throws IOException
     */
    public static String addField(String json, String key, String value) {
        if (StringUtil.isBlank(key)) {
            return json;
        }
        try {
            JsonNode jsonNode = objectMapper.readTree(json);
            ObjectNode objectNode = (ObjectNode) jsonNode;
            objectNode.put(key, value);
            return objectMapper.writeValueAsString(objectNode);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }


    /**
     * 解析列表
     *
     * @param json          原始数据
     * @param arrayPathList 解析数组的路径
     * @param consumer      自定义解析方法
     * @return
     */
    public static String parseArray(String json, List<String> arrayPathList, Consumer<JsonNode> consumer) {
        if (StringUtil.isBlank(json) || CollectionUtil.isEmpty(arrayPathList)) {
            return json;
        }
        try {
            JsonNode jsonNode = objectMapper.readTree(json);
            JsonNode currentNode = jsonNode;
            for (String jsonKey : arrayPathList) {
                if (!currentNode.isObject()) {
                    return json;
                }
                JsonNode subNode = currentNode.get(jsonKey);
                if (null == subNode || subNode.isNull()) {
                    return json;
                }
                currentNode = subNode;
            }

            if (!currentNode.isArray()) {
                //  非数组
                return json;
            }
            ArrayNode arrayNode = (ArrayNode) currentNode;
            for (JsonNode node : arrayNode) {
                consumer.accept(node);
            }
            return objectMapper.writeValueAsString(jsonNode);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }

    }

    /**
     * 获取属性
     *
     * @param json 字符串
     * @param keys key
     * @return
     * @throws IOException
     */
    public static Map<String, String> getField(String json, String... keys) {
        if (ArrayUtil.isEmpty(keys)) {
            return Collections.emptyMap();
        }
        try {
            Map<String, String> resultMap = new HashMap<>();
            JsonNode jsonNode = objectMapper.readTree(json);
            for (String oneKey : keys) {
                JsonNode oldValueNode = jsonNode.findValue(oneKey);
                if (null != oldValueNode && !(oldValueNode instanceof NullNode)) {
                    resultMap.put(oneKey, oldValueNode.asText());
                }
            }
            return resultMap;
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 益处json属性
     *
     * @param json 字符串
     * @param keys key
     */
    public static String removeField(String json, String... keys) {
        if (ArrayUtil.isEmpty(keys)) {
            return json;
        }
        try {
            JsonNode jsonNode = objectMapper.readTree(json);
            ObjectNode objectNode = (ObjectNode) jsonNode;
            for (String oneKey : keys) {
                objectNode.remove(oneKey);
            }
            return objectMapper.writeValueAsString(objectNode);
        } catch (Exception e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * 获取node
     *
     * @param json
     * @return
     */
    public static JsonNode readTree(String json) {
        try {
            JsonNode jsonNode = objectMapper.readTree(json);
            return jsonNode;
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * node转string
     *
     * @param jsonNode
     * @return
     */
    public static String writeValueAsString(JsonNode jsonNode) {
        try {
            return objectMapper.writeValueAsString(jsonNode);
        } catch (JsonProcessingException e) {
            throw new RuntimeException(e);
        }
    }
}
