package cn.com.servyou.utils;

import javax.crypto.Cipher;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import java.security.GeneralSecurityException;

/**
 * 对称加/解密辅助工具类
 *
 * @author luyb@cn.com.servyou.com.cn
 * @version $Id: SymmetricCryptoUtils.java v 0.1 2017/3/16 12:45 luyb Exp $$
 */
public class SymmetricCryptoUtil {

    /** CBC(Cipher Block Chaining:密码分组链接)工作模式 同样的明文分组不一定加密或解密同样的密文块 */
    public static final String CBC          = "CBC";

    /** ECB(Electronic Code Book:电码本) 同样的明文分组总是加密成相同的密文分组 */
    public static final String ECB          = "ECB";

    /** 不填充模式 */
    public static final String NO_PADDING   = "NoPadding";

    /** 每个填充的字节都记录了填充的总字节数 */
    public static final String PKCS5PADDING = "PKCS5Padding";

    /**
     * 对称加密/解密
     *
     * @param data  等待加密/解密数据
     * @param key   密钥数据
     * @param algorithm 加/解密算法
     * @param mode  加/解密模式
     * @return  byte[] 加解密后的数据
     * @throws GeneralSecurityException
     */
    public static byte[] symmetricCrypto(byte[] data, byte[] key, String algorithm,
                                         int mode) throws GeneralSecurityException {
        //算法 工作模式  填充
        String fullAlgorithm = algorithm + "/CBC/PKCS5Padding";
        byte[] iv = initIV(fullAlgorithm);
        return doCrypto(data, key, iv, fullAlgorithm, CBC, PKCS5PADDING, mode);
    }

    /**
     * 实现对称加/解密方法
     *
     * @param data 等待加密/解密数据
     * @param key 密钥数据
     * @param iv 初始化向量，类似salt
     * @param fullAlgorithm 完整加/解密算法
     * @param workingMode   工作模式
     * @param padding 填充模式
     * @param mode  加/解密模式 (cipher.ENCRYPT_MODE；解密——Cipher.DECRYPT_MODE）
     * @return  byte[] 加解密后的数据
     */
    private static byte[] doCrypto(byte[] data, byte[] key, byte[] iv, String fullAlgorithm,
                                   String workingMode, String padding,
                                   int mode) throws GeneralSecurityException {

        if (!CBC.equals(workingMode) && !ECB.equals("ECB"))
            throw new GeneralSecurityException("工作模式只支持CBC和ECB!");

        if (!NO_PADDING.equals(padding) && !PKCS5PADDING.equals(padding))
            throw new GeneralSecurityException("填充模式只支持NoPadding和PKCS5Padding!");

        if (mode != Cipher.ENCRYPT_MODE && mode != Cipher.DECRYPT_MODE) {
            throw new GeneralSecurityException(
                    "错误的加解密标识,目前只支持Cipher.ENCRYPT_MODE和Cipher.DECRYPT_MODE");
        }
        Cipher cipher = getCipher(key, iv, fullAlgorithm, workingMode, padding, mode);
        return cipher.doFinal(data);
    }

    /**
     *
     * @param key 密钥数据
     * @param iv 初始化向量，类似salt
     * @param fullAlgorithm 完整加/解密算法
     * @param workingMode 工作模式(目前接受的参数有CBC和ECB)
     * @param padding 填充模式
     * @param mode 加/解密模式 (cipher.ENCRYPT_MODE；解密——Cipher.DECRYPT_MODE）
     * @return Cipher
     * @throws GeneralSecurityException
     */
    private static Cipher getCipher(byte[] key, byte[] iv, String fullAlgorithm, String workingMode,
                                    String padding, int mode) throws GeneralSecurityException {
        Cipher cipher = Cipher.getInstance(fullAlgorithm);
        SecretKey secretKey = new SecretKeySpec(key,
                substringBefore(fullAlgorithm, "/"));

        if (CBC.equals(workingMode)) {
            IvParameterSpec ivParamSpec = new IvParameterSpec(iv);
            cipher.init(mode, secretKey, ivParamSpec);
        } else {
            cipher.init(mode, secretKey);
        }
        return cipher;
    }

    /**
     * 初始化向量
     *
     * @param fullAlgorithm 完整加密算法(算法+工作模式+填充)
     * @return 初始向量
     * @throws GeneralSecurityException
     */
    private static byte[] initIV(String fullAlgorithm) throws GeneralSecurityException {
        Cipher cipher = Cipher.getInstance(fullAlgorithm);
        //加密模块大小
        int blockSize = cipher.getBlockSize();
        byte[] iv = new byte[blockSize];
        for (int i = 0; i < iv.length; i++) {
            iv[i] = 0;
        }
        return iv;
    }


    public static String substringBefore(String str, String separator) {
        if(!isEmpty(str) && separator != null) {
            if(separator.isEmpty()) {
                return "";
            } else {
                int pos = str.indexOf(separator);
                return pos == -1?str:str.substring(0, pos);
            }
        } else {
            return str;
        }
    }

    public static String substringAfter(String str, String separator) {
        if(isEmpty(str)) {
            return str;
        } else if(separator == null) {
            return "";
        } else {
            int pos = str.indexOf(separator);
            return pos == -1?"":str.substring(pos + separator.length());
        }
    }

    public static boolean isEmpty(CharSequence cs) {
        return cs == null || cs.length() == 0;
    }

}